<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'MWT_Fields_Post' ) ) :

	class MWT_Fields_Post extends MWT_Fields {

		public $mwt_style                = '';
		public $mwt_post_type            = 'post';
		public $mwt_options_subdirectory = 'posts';
		public $mwt_options;
		public $mwt_nonce_created = false;

		/**
		 * __construct
		 *
		 * Adds needed actions.
		 *
		 * @date    7/01/2019
		 *
		 * @param void
		 *
		 * @return    void
		 * @since    0.0.1
		 *
		 */
		public function __construct() {

			// init on post edit screens
			add_action( 'load-post.php', array( $this, 'init' ) );
			add_action( 'load-post-new.php', array( $this, 'init' ) );

			// save
			add_action( 'save_post', array( $this, 'save_post' ), 10, 2 );
		}


		/**
		 * init
		 *
		 * Sets up Form functionality on load post action.
		 *
		 * @date    7/01/2019
		 *
		 * @param void
		 *
		 * @return    void
		 * @since    0.0.1
		 *
		 */
		public function init() {

			// globals
			global $typenow;

			// restrict specific post types
			$mwt_restricted = array( 'attachment' );
			if ( in_array( $typenow, $mwt_restricted, true ) ) {
				return;
			}

			//where options file are located
			$this->mwt_options_directory = apply_filters( 'seedone_options_directory', 'options' );
			$this->mwt_post_type         = $typenow;
			$this->mwt_options_filename  = $typenow;
			$this->mwt_options           = $this->get_fields( $typenow );

			// actions
			add_action( 'add_meta_boxes', array( $this, 'render_meta_boxes' ), 10, 2 );
		}


		/**
		 * render_meta_boxes
		 *
		 * Adds metaboxes for the given $mwt_post_type and $mwt_post.
		 *
		 * @date    7/01/2019
		 *
		 * @param string $mwt_post_type The post type.
		 * @param WP_Post $mwt_post The post being edited.
		 *
		 * @return    void
		 * @since    0.0.1
		 *
		 */
		public function render_meta_boxes( $mwt_post_type, $mwt_post ) {

			// Get options for this post_type.
			$mwt_options = $this->mwt_options;

			// Adding metaboxes for given options
			if ( $mwt_options ) {
				foreach ( $mwt_options as $mwt_metabox_id => $mwt_metabox ) {

					$mwt_title    = $mwt_metabox['title'];                // Metabox title
					$mwt_context  = ! empty( $mwt_metabox['context'] ) ? $mwt_metabox['context'] : 'normal';     // normal, side
					$mwt_priority = ! empty( $mwt_metabox['priority'] ) ? $mwt_metabox['priority'] : 'normal';   // high, core, default, low

					// Reduce priority for sidebar metaboxes for best position.
					if ( 'side' === $mwt_context ) {
						$mwt_priority = 'core';
					}

					// Add the meta box.
					add_meta_box(
						$mwt_metabox_id,
						$mwt_title,
						array( $this, 'render_meta_box' ),
						$mwt_post_type,
						$mwt_context,
						$mwt_priority,
						array( 'metabox' => $mwt_metabox )
					);
				}
			}
		}

		/**
		 * render_meta_box
		 *
		 * Renders metabox fields.
		 *
		 * @date    7/01/2019
		 *
		 * @param WP_Post $mwt_post The post being edited.
		 * @param array metabox The meta box args.
		 *
		 * @return    void
		 * @since    0.0.1
		 *
		 */
		public function render_meta_box( $mwt_post, $mwt_metabox ) {

			if ( ! $this->mwt_nonce_created ) {
				//create our custom nonce field
				wp_nonce_field( - 1, 'seedone-fields' );
				$this->mwt_nonce_created = true;
			}

			foreach ( $mwt_metabox['args']['metabox']['fields'] as $mwt_id => $mwt_option ) {
				$mwt_meta_value = get_post_meta( $mwt_post->ID, $mwt_id, true );
				$mwt_value      = $this->get_value( $mwt_meta_value, $mwt_option );
				$this->render_field( $mwt_id, $mwt_option, $mwt_value, 'div' );
			}
		}

		/**
		 * allow_save_post
		 *
		 * Checks if the $mwt_post is allowed to be saved.
		 * Used to avoid triggering "acf/save_post" on dynamically created posts during save.
		 *
		 * @date    7/01/2019
		 *
		 * @param WP_Post $mwt_post The post to check.
		 *
		 * @return    bool
		 * @since    0.0.1
		 *
		 */
		public function allow_save_post( $mwt_post ) {

			// vars
			$mwt_allow = true;

			// restrict post types
			$mwt_restrict = array( 'auto-draft', 'revision' );
			if ( in_array( $mwt_post->post_type, $mwt_restrict, true ) ) {
				$mwt_allow = false;
			}

			// disallow if the $_POST ID value does not match the $mwt_post->ID
			$mwt_form_post_id = ! empty( $_POST['post_ID'] ) ? (int) $_POST['post_ID'] : false;
			if ( $mwt_form_post_id && $mwt_form_post_id !== $mwt_post->ID ) {
				$mwt_allow = false;
			}

			// revision (preview)
			if ( 'revision' === $mwt_post->post_type ) {

				// allow if doing preview and this $mwt_post is a child of the $_POST ID
				if ( ! empty( $_POST['wp-preview'] ) ) {
					if ( 'dopreview' === $_POST['wp-preview'] && $mwt_form_post_id === $mwt_post->post_parent ) {
						$mwt_allow = true;
					}
				}
			}

			// return
			return $mwt_allow;
		}

		/**
		 * save_post
		 *
		 * Triggers during the 'save_post' action to save the $_POST data.
		 *
		 * @date    7/01/2019
		 *
		 * @param int $mwt_post_id The post ID
		 * @param WP_POST $mwt_post the post object.
		 *
		 * @return    int
		 * @since    0.0.1
		 *
		 */

		public function save_post( $mwt_post_id, $mwt_post ) {

			// bail ealry if no allowed to save this post type
			if ( ! $this->allow_save_post( $mwt_post ) ) {
				return $mwt_post_id;
			}
			// verify nonce
			if ( empty( $_POST['seedone-fields'] ) ) {
				return $mwt_post_id;
			}
			if ( ! wp_verify_nonce( sanitize_title( wp_unslash( $_POST['seedone-fields'] ) ) ) ) {
				return $mwt_post_id;
			}

			//options
			$mwt_options = $this->mwt_options;
			if ( ! empty( $mwt_options ) ) :

				$mwt_keys = array();

				//getting all option keys for save as a post meta
				foreach ( $mwt_options as $mwt_metabox_id => $mwt_metabox ) {
					if ( ! empty( $mwt_metabox['fields'] ) ) {
						$mwt_keys[ $mwt_metabox_id ] = array_merge( array_keys( $mwt_metabox['fields'] ), $mwt_keys );
					}
				}

				// save in post meta
				foreach ( $mwt_keys as $mwt_metabox_id => $mwt_metabox_keys ) :
					foreach ( $mwt_metabox_keys as $mwt_key ) :
						//TODO progress checkbox type here - can be unset
						if ( isset( $_POST[ $mwt_key ] ) ) {

							$mwt_value = sanitize_text_field( wp_unslash( $_POST[ $mwt_key ] ) );
							//for checkbox - set true to 1
							if ( 'checkbox' === $mwt_options[ $mwt_metabox_id ]['fields'][ $mwt_key ]['type'] ) {
								$mwt_value = '1';
							}
							update_post_meta( $mwt_post_id, $mwt_key, $mwt_value );

						}
						//checking empty value for checkbox and putting '0' string if checkbox was not checked
						if ( 'checkbox' === $mwt_options[ $mwt_metabox_id ]['fields'][ $mwt_key ]['type'] && ! isset( $_POST[ $mwt_key ] ) ) {
							update_post_meta( $mwt_post_id, $mwt_key, '0' );
						}
					endforeach;
				endforeach;

			endif; //$mwt_options

			return $mwt_post_id;
		}
	}

endif; //class_exists

new MWT_Fields_Post();
