<?php
/**
 * Widget API: MWT_Widget_Theme_Posts class
 *
 * based on WordPress recent posts
 *
 * @package Seedone
 * @since 0.0.1
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if ( ! class_exists( 'MWT_Widget_Theme_Posts' ) ) :

	class MWT_Widget_Theme_Posts extends WP_Widget {

		/**
		 * Sets up a new Posts widget instance.
		 *
		 * @since 0.0.1
		 */
		public function __construct() {
			$mwt_widget_ops = array(
				'classname'                   => 'widget_custom_posts',
				'description'                 => esc_html__( 'Posts in various layouts.', 'mwt-widgets' ),
				'customize_selective_refresh' => true,
			);
			parent::__construct( 'custom-posts', esc_html__( 'Theme Blog Posts', 'mwt-widgets' ), $mwt_widget_ops );
			$this->alt_option_name = 'widget_custom_posts';
		}

		/**
		 * Outputs the content for the current Custom Posts widget instance.
		 *
		 * @param array $mwt_args Display arguments including 'before_title', 'after_title',
		 *                        'before_widget', and 'after_widget'.
		 * @param array $mwt_instance Settings for the current Custom Posts widget instance.
		 */
		public function widget( $mwt_args, $mwt_instance ) {
			if ( ! isset( $mwt_args['widget_id'] ) ) {
				$mwt_args['widget_id'] = $this->id;
			}

			$mwt_title = ( ! empty( $mwt_instance['title'] ) ) ? $mwt_instance['title'] : '';

			/** This filter is documented in wp-includes/widgets/class-wp-widget-pages.php */
			$mwt_title = apply_filters( 'widget_title', $mwt_title, $mwt_instance, $this->id_base );

			$mwt_sub_title = ( ! empty( $mwt_instance['sub_title'] ) ) ? $mwt_instance['sub_title'] : '';

			$mwt_number = ( ! empty( $mwt_instance['number'] ) ) ? absint( $mwt_instance['number'] ) : 5;
			// May be not integer
			if ( empty( $mwt_number ) ) {
				$mwt_number = 5;
			}

			$mwt_category     = ( ! empty( $mwt_instance['category'] ) ) ? absint( $mwt_instance['category'] ) : '';
			$mwt_show_cat     = isset( $mwt_instance['show_cat'] ) ? $mwt_instance['show_cat'] : false;
			$mwt_layout       = ( ! empty( $mwt_instance['layout'] ) ) ? esc_attr( $mwt_instance['layout'] ) : 'default';
			$mwt_gap          = ( ! empty( $mwt_instance['gap'] ) ) ? esc_attr( $mwt_instance['gap'] ) : '';
			$mwt_show_date    = isset( $mwt_instance['show_date'] ) ? $mwt_instance['show_date'] : false;
			$mwt_text_center  = isset( $mwt_instance['text_center'] ) ? $mwt_instance['text_center'] : false;
			$mwt_css_class    = ( ! empty( $mwt_instance['css_class'] ) ) ? sanitize_text_field( $mwt_instance['css_class'] ) : '';
			$mwt_read_all     = ( ! empty( $mwt_instance['read_all'] ) ) ? sanitize_text_field( $mwt_instance['read_all'] ) : '';
			$mwt_post__not_in = ( ! empty( $mwt_instance['post__not_in'] ) ) ? (array) ( $mwt_instance['post__not_in'] ) : array();

			//layout may contain columns count separated by space
			$mwt_layout  = explode( ' ', $mwt_layout );
			$mwt_columns = ( ! empty( $mwt_layout[1] ) ) ? absint( $mwt_layout[1] ) : 2;
			$mwt_layout  = $mwt_layout[0];

			/**
			 * Filters the arguments for the Recent Posts widget.
			 *
			 * @param array $mwt_args An array of arguments used to retrieve the recent posts.
			 * @param array $mwt_instance Array of settings for the current widget.
			 *
			 * @see WP_Query::get_posts()
			 *
			 */
			// for sticky posts
			// $mwt_sticky_posts = get_option('sticky_posts');
			$mwt_r = new WP_Query(
				array(
					'posts_per_page'      => $mwt_number,
					'no_found_rows'       => true,
					'post_status'         => 'publish',
					'ignore_sticky_posts' => true,
					'cat'                 => $mwt_category,
					'post__not_in'        => $mwt_post__not_in,
				)
			);

			if ( ! $mwt_r->have_posts() ) {
				return;
			}

			$mwt_cat_name = '';
			if ( ! empty( $mwt_category ) ) {
				$mwt_category = get_category( $mwt_category );
				$mwt_cat_name = $mwt_category->cat_name;
			}

			$mwt_read_all_url = '';
			if ( ! empty( $mwt_read_all ) ) {
				if ( ! empty( $mwt_category ) ) {
					$mwt_read_all_url = get_category_link( $mwt_category );
				} else {
					$mwt_read_all_url = get_post_type_archive_link( 'post' );
				}
			}

			$mwt_filepath = MWT_WIDGETS_PLUGIN_PATH . 'widgets/posts/views/' . $mwt_layout . '.php';

			if ( file_exists( $mwt_filepath ) ) {
				include $mwt_filepath;
			} else {
				esc_html_e( 'View not found: ', 'mwt-widgets' );
				echo '<strong>' . esc_html( $mwt_filepath ) . '</strong>';
			}

		}

		/**
		 * Handles updating the settings for the current Recent Posts widget instance.
		 *
		 * @param array $mwt_new_instance New settings for this instance as input by the user via
		 *                            WP_Widget::form().
		 * @param array $mwt_old_instance Old settings for this instance.
		 *
		 * @return array Updated settings to save.
		 */
		public function update( $mwt_new_instance, $mwt_old_instance ) {
			$mwt_instance                = $mwt_old_instance;
			$mwt_instance['title']       = sanitize_text_field( $mwt_new_instance['title'] );
			$mwt_instance['sub_title']   = sanitize_text_field( $mwt_new_instance['sub_title'] );
			$mwt_instance['number']      = (int) $mwt_new_instance['number'];
			$mwt_instance['category']    = (int) $mwt_new_instance['category'];
			$mwt_instance['show_cat']    = isset( $mwt_new_instance['show_cat'] ) ? (bool) $mwt_new_instance['show_cat'] : false;
			$mwt_instance['layout']      = esc_attr( $mwt_new_instance['layout'] );
			$mwt_instance['gap']         = esc_attr( $mwt_new_instance['gap'] );
			$mwt_instance['show_date']   = isset( $mwt_new_instance['show_date'] ) ? (bool) $mwt_new_instance['show_date'] : false;
			$mwt_instance['text_center'] = isset( $mwt_new_instance['text_center'] ) ? (bool) $mwt_new_instance['text_center'] : false;
			$mwt_instance['read_all']    = sanitize_text_field( $mwt_new_instance['read_all'] );
			$mwt_instance['css_class']   = sanitize_text_field( $mwt_new_instance['css_class'] );

			return $mwt_instance;

			//TODO read all (category or post type archive)
		}

		/**
		 * Outputs the settings form for the Recent Posts widget.
		 *
		 * @param array $mwt_instance Current settings.
		 */
		public function form( $mwt_instance ) {
			$mwt_title       = isset( $mwt_instance['title'] ) ? esc_attr( $mwt_instance['title'] ) : '';
			$mwt_sub_title   = isset( $mwt_instance['sub_title'] ) ? esc_attr( $mwt_instance['sub_title'] ) : '';
			$mwt_number      = isset( $mwt_instance['number'] ) ? absint( $mwt_instance['number'] ) : 5;
			$mwt_category    = isset( $mwt_instance['category'] ) ? absint( $mwt_instance['category'] ) : '';
			$mwt_show_cat    = isset( $mwt_instance['show_cat'] ) ? (bool) $mwt_instance['show_cat'] : false;
			$mwt_layout      = isset( $mwt_instance['layout'] ) ? esc_attr( $mwt_instance['layout'] ) : 'default';
			$mwt_gap         = isset( $mwt_instance['gap'] ) ? esc_attr( $mwt_instance['gap'] ) : '';
			$mwt_show_date   = isset( $mwt_instance['show_date'] ) ? (bool) $mwt_instance['show_date'] : false;
			$mwt_text_center = isset( $mwt_instance['text_center'] ) ? (bool) $mwt_instance['text_center'] : false;
			$mwt_css_class   = isset( $mwt_instance['css_class'] ) ? sanitize_text_field( $mwt_instance['css_class'] ) : '';
			$mwt_read_all    = isset( $mwt_instance['read_all'] ) ? sanitize_text_field( $mwt_instance['read_all'] ) : '';
			?>
			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'mwt-widgets' ); ?></label>
				<input
					class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>"
					type="text"
					value="<?php echo esc_attr( $mwt_title ); ?>"/>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'sub_title' ) ); ?>"><?php esc_html_e( 'Sub Title:', 'mwt-widgets' ); ?></label>
				<textarea
					class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'sub_title' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'sub_title' ) ); ?>"
				><?php echo esc_attr( $mwt_sub_title ); ?></textarea>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'category' ) ); ?>"><?php esc_html_e( 'Select category to show:', 'mwt-widgets' ); ?></label>
				<?php
					wp_dropdown_categories(
						array(
							'id'              => $this->get_field_id( 'category' ),
							'name'            => $this->get_field_name( 'category' ),
							'selected'        => $mwt_category,
							'show_option_all' => esc_html__( 'All', 'mwt-widgets' ),
							'hierarchical'    => 1,
							'show_count'      => 1,
							'class'           => 'widefat',
						)
					);
				?>
			</p>

			<p>
				<input
					class="checkbox"
					type="checkbox"<?php checked( $mwt_show_cat ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'show_cat' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'show_cat' ) ); ?>"/>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'show_cat' ) ); ?>"><?php esc_html_e( 'Show category name if selected', 'mwt-widgets' ); ?></label>
			</p>

			<p>
				<?php esc_html_e( 'Any layout in main sidebar will be displayed in one column:', 'mwt-widgets' ); ?>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'layout' ) ); ?>"><?php esc_html_e( 'Widgets layout:', 'mwt-widgets' ); ?></label>
				<select
					name="<?php echo esc_attr( $this->get_field_name( 'layout' ) ); ?>"
					id="<?php echo esc_attr( $this->get_field_id( 'layout' ) ); ?>"
					class="widefat"
				>
					<option
						value="default"<?php selected( $mwt_layout, 'default' ); ?>><?php esc_html_e( 'Default list', 'mwt-widgets' ); ?></option>
					<option
						value="title-only"<?php selected( $mwt_layout, 'title-only' ); ?>><?php esc_html_e( 'Only titles', 'mwt-widgets' ); ?></option>
					<option
						value="featured-columns"<?php selected( $mwt_layout, 'featured-columns' ); ?>><?php esc_html_e( 'Large first post - layout 1', 'mwt-widgets' ); ?></option>
					<option
						value="featured"<?php selected( $mwt_layout, 'featured' ); ?>><?php esc_html_e( 'Large first post - layout 2', 'mwt-widgets' ); ?></option>
					<option
						value="featured-3"<?php selected( $mwt_layout, 'featured-3' ); ?>><?php esc_html_e( 'Large two first posts', 'mwt-widgets' ); ?></option>
					<option
						value="cols"<?php selected( $mwt_layout, 'cols' ); ?>><?php esc_html_e( 'Grid - 2 columns', 'mwt-widgets' ); ?></option>
					<option
						value="cols 3"<?php selected( $mwt_layout, 'cols 3' ); ?>><?php esc_html_e( 'Grid - 3 columns', 'mwt-widgets' ); ?></option>
					<option
						value="cols 4"<?php selected( $mwt_layout, 'cols 4' ); ?>><?php esc_html_e( 'Grid - 4 columns', 'mwt-widgets' ); ?></option>
					<option
						value="cols-absolute-single"<?php selected( $mwt_layout, 'cols-absolute-single' ); ?>><?php esc_html_e( '1 column - title overlap', 'mwt-widgets' ); ?></option>
					<option
						value="cols-absolute"<?php selected( $mwt_layout, 'cols-absolute' ); ?>><?php esc_html_e( 'Grid - 2 cols - title overlap', 'mwt-widgets' ); ?></option>
					<option
						value="cols-absolute 3"<?php selected( $mwt_layout, 'cols-absolute 3' ); ?>><?php esc_html_e( 'Grid - 3 cols - title overlap', 'mwt-widgets' ); ?></option>
					<option
						value="cols-absolute 4"<?php selected( $mwt_layout, 'cols-absolute 4' ); ?>><?php esc_html_e( 'Grid - 4 cols - title overlap', 'mwt-widgets' ); ?></option>
					<option
						value="side"<?php selected( $mwt_layout, 'side' ); ?>><?php esc_html_e( 'Side featured image', 'mwt-widgets' ); ?></option>
				</select>
			</p>


			<p>
				<?php esc_html_e( 'Gap size for columns layout:', 'mwt-widgets' ); ?>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'gap' ) ); ?>"><?php esc_html_e( 'Columns gap:', 'mwt-widgets' ); ?></label>
				<select
					name="<?php echo esc_attr( $this->get_field_name( 'gap' ) ); ?>"
					id="<?php echo esc_attr( $this->get_field_id( 'gap' ) ); ?>"
					class="widefat"
				>
					<option
						value=""<?php selected( $mwt_gap, '' ); ?>><?php esc_html_e( 'None', 'mwt-widgets' ); ?></option>
					<option
						value="1"<?php selected( $mwt_gap, '1' ); ?>><?php esc_html_e( '1px', 'mwt-widgets' ); ?></option>
					<option
						value="2"<?php selected( $mwt_gap, '2' ); ?>><?php esc_html_e( '2px', 'mwt-widgets' ); ?></option>
					<option
						value="3"<?php selected( $mwt_gap, '3' ); ?>><?php esc_html_e( '3px', 'mwt-widgets' ); ?></option>
					<option
						value="4"<?php selected( $mwt_gap, '4' ); ?>><?php esc_html_e( '4px', 'mwt-widgets' ); ?></option>
					<option
						value="5"<?php selected( $mwt_gap, '5' ); ?>><?php esc_html_e( '5px', 'mwt-widgets' ); ?></option>
					<option
						value="10"<?php selected( $mwt_gap, '10' ); ?>><?php esc_html_e( '10px', 'mwt-widgets' ); ?></option>
					<option
						value="15"<?php selected( $mwt_gap, '15' ); ?>><?php esc_html_e( '15px', 'mwt-widgets' ); ?></option>
					<option
						value="20"<?php selected( $mwt_gap, '20' ); ?>><?php esc_html_e( '20px', 'mwt-widgets' ); ?></option>
					<option
						value="30"<?php selected( $mwt_gap, '30' ); ?>><?php esc_html_e( '30px', 'mwt-widgets' ); ?></option>
					<option
						value="40"<?php selected( $mwt_gap, '40' ); ?>><?php esc_html_e( '40px', 'mwt-widgets' ); ?></option>
					<option
						value="50"<?php selected( $mwt_gap, '50' ); ?>><?php esc_html_e( '50px', 'mwt-widgets' ); ?></option>
					<option
						value="60"<?php selected( $mwt_gap, '60' ); ?>><?php esc_html_e( '60px', 'mwt-widgets' ); ?></option>
				</select>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'Number of posts (5 by default):', 'mwt-widgets' ); ?></label>
				<input
					class="tiny-text"
					id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>"
					type="number"
					step="1"
					min="1"
					value="<?php echo esc_attr( $mwt_number ); ?>"
					size="3"/>
			</p>

			<p>
				<input
					class="checkbox"
					type="checkbox"<?php checked( $mwt_show_date ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'show_date' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'show_date' ) ); ?>"/>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'show_date' ) ); ?>"><?php esc_html_e( 'Display post date?', 'mwt-widgets' ); ?></label>
			</p>

			<p>
				<input
					class="checkbox"
					type="checkbox"<?php checked( $mwt_text_center ); ?>
					id="<?php echo esc_attr( $this->get_field_id( 'text_center' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'text_center' ) ); ?>"/>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'text_center' ) ); ?>"><?php esc_html_e( 'Center alignment', 'mwt-widgets' ); ?></label>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'read_all' ) ); ?>"><?php esc_html_e( '\'Read All\' link text', 'mwt-widgets' ); ?></label>
				<input
					class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'read_all' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'read_all' ) ); ?>"
					type="text"
					value="<?php echo esc_attr( $mwt_read_all ); ?>"/>
			</p>

			<p>
				<label
					for="<?php echo esc_attr( $this->get_field_id( 'css_class' ) ); ?>"><?php esc_html_e( 'Custom CSS class:', 'mwt-widgets' ); ?></label>
				<input
					class="widefat"
					id="<?php echo esc_attr( $this->get_field_id( 'css_class' ) ); ?>"
					name="<?php echo esc_attr( $this->get_field_name( 'css_class' ) ); ?>"
					type="text"
					value="<?php echo esc_attr( $mwt_css_class ); ?>"/>
			</p>
			<?php
		}
	}

endif; //class_exists

if ( ! function_exists( 'seedone_theme_register_widget_theme_posts' ) ) :
	function seedone_theme_register_widget_theme_posts() {
		register_widget( 'MWT_Widget_Theme_Posts' );
	}
endif;
add_action( 'widgets_init', 'seedone_theme_register_widget_theme_posts' );
